######################################################
##
##  Parametric bootstrap of nested regression models
##
##  Supports testing of fixed and random effects of 
##  lm, glm, lmer, and glmer models.
##
##  December 7, 2015 Mark Banghart
##
##
######################################################

#
#  update documentation to reflect multi-core info
#
#  Look as using lme4 lower level functions to identify
#  random terms
# 


#' @import parallel 
NULL

#' @import lme4 
NULL


# split a list of random variables into 
# individual variables to be estimated
#
# "#" is used to denote covariance
#
varList <- function (varStr) {
  if (grepl("[-] *1",varStr)) {
    # no intercept
    vars <- gsub("[-] *1","",varStr)
    vars <- gsub(" *[+] *"," ",vars)
    vars <- strsplit(vars," ")[[1]]
  } else {
    vars <- gsub(" *[+] *"," ",varStr)
    vars <- strsplit(vars," ")[[1]]
    if (!("1" %in% vars)) {
      # implied "(intercept)"
      vars <- c("1",vars)
    }
    vars <- ifelse(vars == "1","(Intercept)",vars)
  }
  vars <- vars[(vars != "")]
  if (length(vars) > 1) {
    for (i in 1:(length(vars) - 1)) {
      for (j in (i + 1):length(vars))
        vars <- c(vars,paste(vars[i],vars[j],sep="#"))
    }
  }
  vars
}

# split a random term into the individual terms estimated
#
rSplit <- function(rt) {
  sTerm <- strsplit(rt,"\\|")[[1]]
  grp <- gsub(" ", "",sTerm[2], fixed = TRUE)
  vars <- varList(sTerm[1])
  vt <- c("")[-1]
  for (v in vars) {
    vt <- c(vt,paste(v," | ",grp,sep=""))
  }
  vt
}

# terms list with individual random terms
#
rTerms <- function(terms) {
  rInd <- (grepl("\\|",terms))
  fix <- terms[!rInd]
  ran <- terms[rInd]
  if (length(ran) == 0) return(terms)
  rts <- c("")[-1]
  for (rt in ran) {
    rts <- c(rts,rSplit(rt))
  }
  c(fix,rts)
}

# returns the index for a coefficient or variance
#
termInd <- function(mod,term) {
  if (grepl("\\|",term)) {
    sTerm <- strsplit(term,"\\|")[[1]]
    grp <- gsub(" ", "",sTerm[2], fixed = TRUE)
    var <- gsub(" ", "",sTerm[1], fixed = TRUE)[1]
    if (grepl("[#]",var)) {
      corr <- TRUE
      var <- strsplit(var,"\\#")[[1]]
    } else {
      corr <- FALSE
    }

    vc <- lme4::VarCorr(mod)
    vcGrps <- names(vc)
    front <- substr(vcGrps,1,nchar(grp))
    back <- substr(vcGrps,(nchar(grp) + 1),10000)
    gDup <- (nchar(back) == 0 |
               (nchar(back) == 2 &
                  grepl("[.][[:digit:]]",substr(back,1,2))
               )
    )
    gInds <- which( front == grp & gDup)
    for (gi in gInds) {
      giVars <- names(attr(vc[[gi]],"stddev"))
      if (sum(!(var %in% giVars)) == 0) {
        # all var in giVars
        vInds <- which(giVars %in% var)
        if (corr) {
          vInd <- max(vInds)
          cInd <- min(vInds)
        } else {
          vInd <- vInds[1]
          cInd <- NA
        }
        gInd <- gi
      }
    }
    fInd <- NA
  } else {
    fInd <- which(rownames(coefficients(summary(mod))) == term)
    if (length(fInd) == 0) {
      fInd <- NA
    }
    gInd <- vInd <- cInd <- NA
  }

  list(fInd=fInd,gInd=gInd,vInd=vInd,cInd=cInd)
}

# THIS FUNCTION NEEDS TO BE  RENAMED MAYBE - evalMods ---------------------------------------------

# Gets the test statistic from two nested models
#
testNM <- function(test,parmInd,nestStat=TRUE) {
  parm <- NA
  parFrame <- parent.frame()

  stat <- NA
  if (nestStat) {
    classM0 <- eval(quote(class(m0)),envir=parFrame)
    m0MerMod <- eval(quote(inherits(m0,"merMod")),envir=parFrame)
    classM1 <- eval(quote(class(m1)),envir=parFrame)
    m1MerMod <- eval(quote(inherits(m1,"merMod")),envir=parFrame)
    if (classM0[1] == "failure" | classM1[1] == "failure") {
      return(list(stat=NA,parm=NA))
    }
    if (m1MerMod){
      if (m0MerMod){
        anovaText <- paste0('anova(m0,m1,test="',test,
                            '",refit=FALSE)$"Chisq"[2]')
      } else {
        # merMod anova reverses order
        anovaText <- paste0('anova(m1,m0,test="',test,
                            '",refit=FALSE)$"Chisq"[2]')
      }
    } else {
      if (classM1[1] == "lm"){
        anovaText <- paste0('anova(m0,m1,test="',test,'")$"Sum of Sq"[2]')
      } else {
        anovaText <- paste0('anova(m0,m1,test="',test,'")$Deviance[2]')
      }
    }

    stat <- eval(parse(text=anovaText),envir=parFrame)
  }

  if (!is.na(parmInd$fInd)) {
    m1Coef <- eval(quote(summary(m1)$coefficients),envir=parFrame)
    parm <- m1Coef[parmInd$fInd,"Estimate"]
  } else if (!is.na(parmInd$gInd)) {
    m1VarCorr <- eval(quote(VarCorr(m1)),envir=parFrame)
    gi <- m1VarCorr[[parmInd$gInd]]
    if (is.na(parmInd$cInd)) {
      parm <- gi[parmInd$vInd,parmInd$vInd]
    } else {
      parm <- gi[parmInd$vInd,parmInd$cInd]
    }
  }
  (list(stat=stat,parm=parm))
}


# run expression returning results and warning indicator
#
areWarnings <- function(expr) {
  warn <- FALSE
  warnEnv <- environment()
  warnTrue <- function(w) {
    assign("warn", TRUE,
           envir = warnEnv)
  }
  list(resp=withCallingHandlers(expr, warning=warnTrue),
       warn= warn)
}


# Refit a model
#
reModel <- function(modName,modEnvir,sim1Envir) {
  if (eval(parse(text=paste0("inherits(",modName,',"merMod")')),
           envir=modEnvir)) {
    call <- eval(parse(text=paste0("getCall(",modName,")")),envir=modEnvir)
    call$formula <- update.formula(formula(call), "sim~.")
    callExp <- parse(text=deparse(call))
    assign(modName,eval(callExp,envir=modEnvir),envir=sim1Envir)
  } else {
    call <- eval(parse(text=paste0("getCall(",modName,")")),envir=modEnvir)
    call$formula <- update.formula(formula(call), "sim~.")
    callExp <- parse(text=deparse(call))
    assign(modName,eval(callExp,envir=modEnvir),envir=sim1Envir)
  }
  NA
}


# Single simulation run of LR for nested models
#
sim1NM <- function(test,parmInd,modEnvir=NULL,nestStat=TRUE) {
  errAlt <- FALSE
  warnAlt <- FALSE
  errNull <- FALSE
  warnNull <- FALSE
  sim1Envir <- environment()

  # m0 and m1 are assigned object in this environment in reModel
  if (nestStat) {
    r0 <- try(areWarnings(reModel("m0",modEnvir,sim1Envir)),silent=TRUE)
    if (class(r0) == "try-error") {
      m0 <- NA
      class(m0) <- "failure"
      errNull <- TRUE
    } else {
      warnNull <- r0$warn
    }
  }

  r1 <- try(areWarnings(reModel("m1",modEnvir,sim1Envir)),silent=TRUE)
  if (class(r1) == "try-error") {
    m1 <- NA
    class(m1) <- "failure"
    errAlt <- TRUE
  } else {
    warnAlt <- r1$warn
  }
  val <- testNM(test,parmInd,nestStat)
  c(stat=val$stat,parm=val$parm,errAlt=errAlt,errNull=errNull,
    warnAlt=warnAlt,warnNull=warnNull)
}


# Simulation runs a tasks portion of the runs
#
simRNM <- function(X,m1,m0,nsimTask=1000,parmInd, # seed,
                   test="LRT",progress=NULL,mdf,nestStat=TRUE) {
  RNGkind("L'Ecuyer-CMRG")
  coreId <- X$coreId
  assign(".Random.seed", X$seed, envir=.GlobalEnv)
  if (!is.null(progress)) {
    logFile <- paste0("pbnmLog_",coreId,".txt")
    cat("Task",coreId,"started with seed",.Random.seed,"at", date(),"\n",
        file = logFile,append = TRUE)
  }
  dist <- matrix(rep(NA, (nsimTask * 6)), ncol=nsimTask)
  rownames(dist) <- c("stat","parm","errAlt","errNull",
                      "warnAlt","warnNull")

  envir <- environment()
  for (i in 1:nsimTask) {
    sim <- simulate(m0)[,1]
    if (sim[1] == 2) {
      # This test is only to avoide a lint warning about sim not being used
    }
    dist[,i] <- sim1NM(test,parmInd,modEnvir=envir,nestStat=nestStat)
    if (!is.null(progress) ) {
      if (i %% progress == 0) {
        cat("Task",coreId,"has completed",i,"simulations at", date(),"\n",
            file = logFile,append = TRUE)
      }
    }
  }
  if (!is.null(progress)) {
    cat("Task",coreId,"complete at", date(),"\n",
        file = logFile,append = TRUE)
  }
  dist
}


# Internal simulation function.  Used by both
# pbnm() and pbnmRefdist()
#
pbnmRD <- function(m1,m0,nsim=1000,cores=1,tasks=1,
                   seed=NULL,test="LRT",progress=NULL,
                   nestStat=TRUE,
                   callEnvir=NULL,matchCall=NULL
                   ) {
  if (is.null(callEnvir)) {
    stop("No calling frame provided")
  }
  if (is.null(matchCall)) {
    stop("No matchCall object provided")
  }

  if ( !( inherits(m1,"merMod") | inherits(m1,"lm") ) ) {
    stop("m1 does not inherit from merMod or lm")
  }
  if ( !( inherits(m0,"merMod") | inherits(m0,"lm") ) ) {
    stop("m0 does not inherit from merMod or lm")
  }
  if (!is.numeric(nsim)) {
    stop("nsim must be integer")
  }
  if (nsim %% 1 != 0) {
    stop("nsim must be integer")
  }
  if (!is.numeric(cores)) {
    stop("cores must be integer")
  }
  if (cores %% 1 != 0) {
    stop("cores must be integer")
  }
  if (!is.numeric(tasks)) {
    stop("tasks must be integer")
  }
  if (tasks %% 1 != 0) {
    stop("tasks must be integer")
  }
  if (!is.null(seed)) {
    if (!is.numeric(seed)) {
      stop("seed must be integer")
    }
    if (seed %% 1 != 0) {
      stop("seed must be integer")
    }
  }
  if (!is.null(progress)) {
    if (!is.numeric(progress)) {
      stop("progress must be integer")
    }
    if (progress %% 1 != 0) {
      stop("progress must be integer")
    }
  }
  if (!(test == "LRT" | test == "F")) {
    stop("test must be F or LRT")
  }
  # determine if any differences in the models are in the fixed terms
  call0 <- getCall(m0)
  call1 <- getCall(m1)
  terms0 <- attr( terms(as.formula(call0[[2]])), "term.labels")
  terms1 <- attr( terms(as.formula(call1[[2]])), "term.labels")
  fixed1 <-  terms1[!grepl("\\|",terms1)]
  fixDiff <- any(!(fixed1 %in% terms0))

  if ( inherits(m1,"merMod") & inherits(m0,"merMod") ) {
    if (lme4::isREML(m0)) {
      if (!lme4::isREML(m1)) {
        stop("m0 is REML and m1 is not")
      } else if (fixDiff) {
        stop(paste0("Models need to be fit by ML when",
                    " there are differences in fixed terms")
            )
      }
    } else {
      if (lme4::isREML(m1)) {
        stop("m1 is REML and m0 is not")
      }
    }
  }
  if (inherits(m1,"merMod")) {
    if (test != "LRT") {
      stop("test must be LRT for mixed models")
    }
    if ( !inherits(m0,"merMod") ) {
      if (lme4::isREML(m1)) {
        stop(paste0("m1 must be fit with MLE if m0 does not",
                    " inherit from merMod"))
      }
    }
  } else if (class(m1)[1] == "lm") {
    if (test != "F") {
      stop("test must be F for linear models")
    }
  } else if (class(m1)[1] == "glm") {
    if (test != "LRT") {
      stop("test must be LRT for generalized models")
    }
  }

  if (!is.numeric(cores)) {
    stop("cores must be numeric")
  }
  if (cores < 1) {
    stop("cores must be greater than equal to 1")
  }
  if (!is.numeric(tasks)) {
    stop("tasks must be numeric")
  }
  if (tasks < 1) {
    stop("tasks must be greater than equal to 1")
  }
  if (tasks < cores) {
    stop("tasks must be greater than or equal to the number of cores")
  }
  if (!is.logical(nestStat)) {
    stop("nestStat must be logical")
  }

  # Check if testing a single dropped term
  terms0 <- rTerms(terms0)
  terms1 <- rTerms(terms1)
  parmInd <- list(fInd=NA,gInd=NA,vInd=NA,cInd=NA)
  if ( sum( !(terms0 %in% terms1)) == 0 ) {
    testTerms <- terms1[!(terms1 %in% terms0)]
    if (length(testTerms) == 1) {
      parmInd <- termInd(m1,testTerms)
    }
  }

  if (!is.na(parmInd$fInd) | !is.na(parmInd$gInd)) {
    if (!nestStat) {
      stop(paste0("nestStat must be TRUE when the difference",
                  "in models is not a single term"))
    }
  }

  tasks <- ceiling(tasks)
  userRNGkind <- RNGkind()
  if (tasks > 1) {
    nsimTask <- ceiling(nsim / tasks)
    RNGkind("L'Ecuyer-CMRG")
    if ( !is.null(seed)) {
      set.seed(seed)
    }
    seeds <- list()
    seeds[[1]] <- list(coreId=1,seed=.Random.seed)
    for (i in 2:tasks) {
      seeds[[i]] <- list(coreId=i,seed=nextRNGStream(seeds[[(i - 1)]]$seed))
    }
  } else {
    nsimTask <- nsim
    if ( !is.null(seed)) {
      set.seed(seed)
    }
  }

  if (cores == 1) {
    dist <- matrix(rep(NA, (nsim * 6)), ncol=nsim)
    rownames(dist) <- c("stat","parm","errAlt","errNull",
                        "warnAlt","warnNull")
    envir <- environment()
    for (i in 1:nsim) {
      if ( (tasks > 1) & (i %% nsimTask == 1) ) {
        tId <- floor(i / nsimTask) + 1
        assign(".Random.seed", seeds[[tId]]$seed, envir=.GlobalEnv)
      }
      sim <- simulate(m0)[,1]
      if (sim[1] == 2) {
        # This test is only to avoide a lint warning about sim not being used
      }
      dist[,i] <- sim1NM(test,parmInd,modEnvir=envir,nestStat=nestStat)
      if (!is.null(progress) ) {
        if (i %% progress == 0) {
          message(i," simulations complete ",Sys.time())
        }
      }
    }
  } else {
    # use multiple cores
    availCores <- parallel::detectCores()
    if (availCores < cores) {
      cores <- availCores
    }
    clus <- parallel::makeCluster(cores,type="PSOCK")
    if (!is.null(progress) ) {
      cat("pbnmRefDist is starting",tasks,"tasks using",
          cores,"cores at", date(),
          "\nSee pbnmLog_n.txt files for progress\n")
    }

    mcl <- matchCall
    class(mcl) <- "list"
    m1Par <- mcl[["m1"]]
    m0Par <- mcl[["m0"]]
    mdf1 <- eval(parse(text=paste0("model.frame(",m1Par,")")),callEnvir)
    mdf0 <- eval(parse(text=paste0("model.frame(",m0Par,")")),callEnvir)
    vars0 <- names(mdf0)
    vars1 <- names(mdf1)
    extVars0 <- vars0[!(vars0 %in% vars1)]
    mdf <- cbind(mdf1,mdf0[,extVars0])
    wInd <- match(c("weights"), names(call1), 0L)
    if (wInd != 0) {
      colnames(mdf)[colnames(mdf) == "(weights)"] <- as.character(call1[[wInd]])
    }
    oInd <- match(c("offset"), names(call1), 0L)
    if (oInd != 0) {
      colnames(mdf)[colnames(mdf) == "(offset)"] <- as.character(call1[[oInd]])
    }

    m0Save <- m0
    m1Save <- m1
    m1Call <- getCall(m1)
    m1Call$data <- quote(mdf)
    if (inherits(m1,"merMod")) {
      m1@call <- m1Call
    } else {
      m1$call <- m1Call
    }
    m0Call <- getCall(m0)
    m0Call$data <- quote(mdf)
    if (inherits(m0,"merMod")) {
      m0@call <- m0Call
    } else {
      m0$call <- m0Call
    }

    parallel::clusterEvalQ(clus, library(pbnm))
    parallel::clusterEvalQ(clus, library(lme4))
    parallel::clusterExport(clus,
                  c("m1","m0","nsimTask","parmInd","test","progress","mdf"),
                  envir=environment()
    )
    dists <- parallel::parLapplyLB(cl=clus,
                         X=seeds,
                         fun=simRNM,
                         m1=m1,m0=m0,nsimTask=nsimTask,parmInd=parmInd,
                         test=test,progress=progress,
                         mdf=mdf
    )
    dist <- do.call(cbind,dists)[,1:nsim]
    m0 <- m0Save
    m1 <- m1Save
    parallel::stopCluster(clus)
  }
  RNGkind(userRNGkind[1]) # restore RNG type
  t(dist)
}


#' Runs a parametric bootstrap testing nested models.
#' 
#' Parametric simulation of new responses, are done using the 
#' \code{\link[stats]{simulate}} function.
#' The nested models are tested on these generated responses
#' using the \code{\link[stats]{anova}} function.
#' A vector is returned with the elements the anova test statistics.
#' If a single estimated parameter is the difference between the two
#' models, a vector of this single parameter is also returned.
#'
#' @param m1,m0 Objects of class lm, glm, lmer, or glmer.
#' @param nsim The number of runs to bootstrap.
#' @param seed An optional parameter. The seed used to start the simulations.
#' @param test to be used by the anova function on the nested models.  
#'   "LRT" and "F" are supported.
#' @param progress An optional parameter.
#'   If set to an integer, a status message will be issued
#'   at that frequency.
#' @param cores An optional parameter.
#'   The number of cores to use.
#' @param tasks An optional parameter.
#'   The number of tasks to split the simulations into.
#' @param nestStat Defaults to \code{\link[base]{TRUE}}.
#'   If set to \code{\link[base]{FALSE}} no nested statistic will
#'   be collected with each simulation.
#' 
#' 
#' @details Error displays are suppressed.
#'   Simulations in which at least one of the models generated an
#'   error are indicated as \code{\link[base]{NA}} in the 
#'   returned stat and paramater vectors.
#'   Simulations in which errors are produced are also indicated 
#'   in the returned errAlt and errNull vectors.
#'
#'   Warnings are displayed.
#'   Simulations in which warning are produced are also indicated 
#'   in the returned warnAlt and warnNull vectors.
#' 
#'   If the difference in the \code{m1} and \code{m0} formula's
#'   is the addition of a single parameter in the \code{m1} formula,
#'   the estimate for this single parameter is recorded for each
#'   simulation.
#'
#'   If there is a single parameter difference between the two models
#'   and the nestStat paramter is set to FALSE, then only the \code{m1}
#'   model will be fit for each simulation.
#'   This can reduce the run time of the boot strap.
#' 
#'   Models which inherit from "meMod" use refit to create the model
#'   for the simulated responses.
#'   Models from lm or glm are created by updating the call formula to
#'   use the simulated response and then the call is evaluated.
#'   The simulated response for each run is generated by a call to
#'   \code{simulate()} using \code{m0} as the reference for the
#'   distribution of each response.
#'   Models generated using weights, offset, or subsets
#'   are supported.  
#'   
#'   A separate Random sequence is used for each of the task.
#'   This allows reproducibility of results when multiple cores are
#'   used.
#'   Multiple tasks can be used with a single core.
#'   
#'   The number of available cores is determined using the
#'   \code{\link[parallel]{detectCores}}  function.
#'   If the number of cores detected is less than the number of cores
#'   requested in the \code{cores} parameter, the number
#'   detected is used.
#'   
#'   The number of tasks should be set to a value larger than the number
#'   of cores which will be used.
#'   If the number of task is smaller, the work is shared only on that
#'   many cores.
#'   Load balancing is uses in sharing the work with the other cores.
#'   So there can be advantages to having tasks be several times the
#'   size of the number of cores.
#'
#'
#' @return A numeric matrix containing the columns 
#'   \describe{
#'     \item{stat}{Value of the test statistic of the nested parameters.
#'       An \code{NA} element indicates that at least one of the models
#'       failed to run.}
#'     \item{parm}{If a single estimated parameter is the difference between
#'       the two models, this column contains the values of this parameter.
#'       Otherwise this column is \code{NA}s.}
#'     \item{errAlt}{An element of this column is \code{TRUE} if at
#'       least one error was issued when m1 was modeled.
#'       Otherwise it is \code{FALSE}.}
#'     \item{errNull}{An element of this column is \code{TRUE} if at
#'       least one error was issued when m0 was modeled.
#'       Otherwise it is \code{FALSE}.}
#'     \item{warnAlt}{An element of this column is \code{TRUE} if
#'       at least one warning was issued when m1 was modeled.
#'       Otherwise it is \code{FALSE}.}
#'     \item{warnNull}{An element of this column is \code{TRUE} if
#'       at least one warning was issued when m0 was modeled.
#'       Otherwise it is \code{FALSE}.}
#'
#'   There is a row for each simulated response.
#' }
#'
#' @examples
#' data(pbDat)
#' mod <- lm(y~x1 + x2 + g1, data=pbDat)
#' modNull <- lm(y~x2 + g1, data=pbDat)
#' ref <- pbnmRefdist(mod,modNull,seed=836928,test="F")
#' obs <- anova(modNull,mod)$"Sum of Sq"[2]
#' mean(abs(ref[,"stat"])>abs(obs))
#' # similar value to Wald test in summary()
#' summary(mod)$coefficients["x1","Pr(>|t|)"]
#'
#' @export
#'
pbnmRefdist <- function(m1,m0,nsim=1000,cores=1,tasks=1,
                        seed=NULL,test="LRT",progress=NULL,
                        nestStat=TRUE
                        ) {
  callEnvir <- parent.frame()
  mc <- match.call()
  pbnmRD(m1=m1,m0=m0,nsim=nsim,cores=cores,tasks=tasks,
         seed=seed,test=test,progress=progress,
         callEnvir=callEnvir,matchCall=mc)
}




#' Run parametric boot strap and return a pbnm object.
#' 
#' Runs the boot strap by calling the pbnmRefdist function.
#' In addition it returns a pbnm object which is accepted by
#' summary, confint, and plot.
#' 
#' @inheritParams pbnmRefdist
#' @param save When \code{TRUE}, save the pbnm object to a file.
#'   This is useful for scripts which include many calls to pbnm.
#' @param fileName Quoted string with the name of the file where
#'   the pbnm object is to be saved.
#' @param pbnmName The name to be given to the pbnm object saved
#'   in the file.
#' @param testContrast Name of the test being done with the
#'   nested models.
#'   This is useful when the test is other than for parameters
#'   being equal to zero.
#' 
#' 
#' @details The pbnm function uses the same parameteric bootstrap
#'   function as the \code{\link{pbnmRefdist}} function.
#'   See the help for \code{\link{pbnmRefdist}} more details on
#'   the parametric bootstrap.
#'   The pbnm object is created from the returned distributions,
#'   warning, and error information.
#' 
#'   If save is \code{TRUE}, the pbnm object can be loading into
#'   an R session using \code{\link[base]{load}}.
#'   The \code{\link[base]{attach}} function can also be used to
#'   access the saved pbnm object.
#' 
#'   If the all model terms of \code{m0} are represented in the 
#'   terms for \code{m1}, the test name is set to the list of 
#'   additional terms in \code{m1}.
#'   If the terms being tested can not be determined from the
#'   terms of the two models, the test name is set to the character
#'   string provided in the \code{testContrast} parameter.
#'   If the \code{testContrast} parameter is not provided,
#'   the test name is then set to indicate that the test is unknown.
#'   
#' @return An object of class "pbnm".
#'   
#'   An object of class "pbnm" is a list containing the following 
#'   components:
#'   
#'   \describe{
#'     \item{nestStat}{\code{TRUE}, indicates that a nested
#'       model statistic was collected in the parametric bootstrap.}
#'     \item{statDist}{Vector of test statistics from simulated
#'       responses.}
#'     \item{statObs}{value of test statistic from the fit of m1 and m0.}
#'     \item{statName}{Name of statistic used in nested model test.}
#'     \item{oneParm}{\code{TRUE},  indicates that the difference in
#'       \code{m1} and \code{m0} is the addition of a single parameter.
#'       If \code{TRUE}, the \code{parmDist}, \code{parmObs}, and
#'       \code{parmName} are returned in the pbnm object.}
#'     \item{parmDist}{Only present if oneParm is \code{TRUE}.
#'       Vector of the dropped parameters value from the
#'       simulated m1 models.}
#'     \item{parmObs}{Only present if oneParm is \code{TRUE}.
#'       Value of the dropped parameters value from the m1 model.}
#'     \item{parmName}{Only present if oneParm is \code{TRUE}.
#'       Name of the dropped parameter.}
#'     \item{errNull}{An element of this column is \code{TRUE} if at
#'       least one error was issued when m0 was modeled.
#'       Otherwise it is \code{FALSE}.}
#'     \item{errAlt}{An element of this column is \code{TRUE} if at
#'       least one error was issued when m1 was modeled.
#'       Otherwise it is \code{FALSE}.}
#'     \item{warnNull}{An element of this column is \code{TRUE} if
#'       at least one warning was issued when m0 was modeled.
#'       Otherwise it is \code{FALSE}.}
#'     \item{warnAlt}{An element of this column is \code{TRUE} if
#'       at least one warning was issued when m1 was modeled.
#'       Otherwise it is \code{FALSE}.}
#'     \item{nsim}{The \code{nsim} parameter value used in the boot strap.}
#'     \item{seed}{The \code{seed} parameter value used in the bootstrap.}
#'     \item{time}{Time and date when the bootstrap was run.}
#'     \item{tested}{A character string which names the is the
#'       terms tested.}
#'     \item{testedDropped}{\code{TRUE} if all of the \code{m0} terms
#'       are in the \code{m1} model.}
#'     \item{mod}{The call from \code{m1}.}
#'     \item{nullMod}{The call from \code{m0}.}
#'   }
#'   
#' @examples
#' data(pbDat)
#' mod <- lm(y~x1 + x2 + g1, data=pbDat)
#' modNull <- lm(y~x2 + g1, data=pbDat)
#' pb <- pbnm(mod,modNull,seed=836928,test="F")
#' summary(pb)
#' 
#' @export 
#' 
pbnm <- function(m1,m0,nsim=1000,seed=NULL,test="LRT",testContrast=NULL,
                 cores=1,tasks=1,progress=NULL,nestStat=TRUE,
                 save=FALSE, fileName=NULL,pbnmName="pbmnResults"
                 ) {
  callEnvir <- parent.frame()
  mc <- match.call()

  # test save, fileName and pbnmName
  if (!is.logical(save)) {
    stop("save must be logical")
  }
  if (!is.null(fileName)) {
    if (!is.character(fileName)) {
      stop("fileName must be type character")
    }
  }
  if (!is.null(pbnmName)) {
    if (!is.character(pbnmName)) {
      stop("pbnmName must be type character")
    }
  }

  obj <- list()
  attr(obj,"heading") <- "Parametric bootstrap of nested models"
  class(obj) <- "pbnm"

  dist <- pbnmRD(m1=m1,m0=m0,nsim=nsim,cores=cores,tasks=tasks,
                 seed=seed,test=test,progress=progress,
                 nestStat=nestStat,
                 callEnvir=callEnvir,matchCall=mc)
  dist <- as.data.frame(dist)

  call0 <- getCall(m0)
  call1 <- getCall(m1)
  terms0 <- attr( terms(as.formula(call0[[2]])), "term.labels")
  terms1 <- attr( terms(as.formula(call1[[2]])), "term.labels")
  terms0 <- rTerms(terms0)
  terms1 <- rTerms(terms1)
  oneParm <- FALSE
  parmInd <- list(fInd=NA,gInd=NA,vInd=NA,cInd=NA)
  if (sum(!(terms0 %in% terms1)) == 0) {
    tested <- terms1[!(terms1 %in% terms0)]
    testedDropped <- TRUE
    if (length(tested) == 1) {
      parmInd <- termInd(m1,tested)
      if (!is.na(parmInd$fInd) | !is.na(parmInd$gInd) ) {
        oneParm <- TRUE
      }
    }
    if(!is.null(testContrast)) {
      warning(paste0("testContrast ignored.",
                     "  The test was identified from model parameters")
             )
    }
  } else{
    if (is.null(testContrast)) {
      tested <- "Variable mismatch between models, unknown test"
    } else {
      if (!is.character(testContrast)) {
        stop("testContrast must be a character string")
      }
      tested <- testContrast
    }
    testedDropped <- FALSE
  }
  obj$nestStat <- nestStat
  observed <- testNM(test,parmInd,nestStat)
  if (nestStat) {
    obj$statDist <- dist$stat
    obj$statObs <- observed$stat
  } else {
    obj$statDist <- NA
    obj$statObs <- NA
  }

  reml <- FALSE
  if ( inherits(m1,"merMod") & inherits(m0,"merMod") ) {
    if (lme4::isREML(m0) & lme4::isREML(m1)) {
      reml <- TRUE
    }
  }
  obj$statName <- if (reml) {
    paste("Restricted",test,"stat")
  } else {
    paste(test,"stat")
  }
  obj$oneParm <- oneParm
  if (oneParm) {
    obj$parmDist <- dist$parm
    obj$parmObs <- observed$parm
    obj$parmName <- tested
  }
  obj$errNull <- dist$errNull
  obj$errAlt <- dist$errAlt
  obj$warnNull <- dist$warnNull
  obj$warnAlt <- dist$warnAlt
  obj$nsim <- nsim
  if (!is.null(seed)) {
    obj$seed <- seed
  }
  obj$time <- date()
  obj$tested <- tested
  obj$testedDropped <- testedDropped
  obj$Mod <- call1
  obj$nullMod <- call0

  if (!save) return(obj)
  if (is.null(fileName)) {
    modName <- deparse(substitute(m1))
    test <- gsub("\\|","_",obj$tested)
    test <- gsub(" ","",test)
    fileName <- paste(modName,test,sep="-")
  }
  nameType <- paste(fileName,".RData",sep="")
  assign(pbnmName,obj)
  save(list=pbnmName,file=nameType)
  obj
}

#' Summarize a parametric bootstrap of nested models
#' 
#' Summary method for class "pbnm"
#' 
#' @param object A "pbnm" object.
#' @param parm Set to \code{TRUE} to summarize the parmDist.
#'   Set to \code{FALSE} to summarize the statDist.
#' @param ... further arguments passed to or from other methods.
#'   
#' @details Returns a p-value for a test from a "pbnm" object
#'   if there were no errors fitting the simulated responses.
#'   If there were fit errors, a range of possible p-values
#'   is returned.
#'   
#'   The number of warnings is also returned.
#'   If the number of are not a small proportion of the number
#'   of simulations,
#'   the p-value is questionable.
#'   When this occurs, it is advised to check the model for issues
#'   which might be causing the warnings (such as small group sizes.)
#' 
#' 
#' @return An object of class "summary.pbnm".
#'   
#'   An object of class "summary.pbnm" is a list containing the
#'   following components:
#'   
#'   \describe{
#'     \item{"P(>=obs)"}{The p-value from the simulated distribution.
#'       Set to \code{NA} if any of the simulations had errors.}
#'     \item{#(>=obs)/runs"}{Only present if "P(>=obs)"==\code{NA}.
#'       The proportion of the number of models simulated
#'       where the simulated response model value is 
#'       greater than or equal to the observed value using m0 and m1.}
#'     \item{"#(>=obs)/models"}{Only present if "P(>=obs)"==\code{NA}.
#'       The proportion of the number of models which did not have
#'       a warning where the simulated response model value is 
#'       greater than or equal to the observed value using m0 and m1.}
#'     \item{"#(>=obs)+fails/runs"}{Only present if "P(>=obs)"==\code{NA}.
#'       The proportion of the number of models simulated
#'       where the simulated response model value is 
#'       greater than or equal to the observed value using m0 and m1
#'       or there was a fit failure.}
#'    \item{noEst}{The number of simulations in which at least one
#'       of the two models had an error or a warning.
#'       These simulation runs are assumed to have an unknown value.}
#'     \item{fails}{The number of simulations in which at least one
#'       of the two models failed.}
#'     \item{warnNull}{The number of simulations in which warning was
#'       issued during the fit of m0.}
#'     \item{warnAlt}{The number of simulations in which warning was
#'       issued during the fit of m1.}
#'     \item{warn}{The number of simulations in which warning was
#'       issued during the fit of either m0 or m1.}
#'     \item{name}{A description of the test which was done in the
#'       simulations.}
#'     \item{observed}{The observed value from m0 and m1.}
#'     \item{testedDropped}{\code{TRUE} if all of the m0 terms are in
#'       the m1 model.}
#'     \item{tested}{A character string which is the terms tested.}
#'     \item{mod}{The call from the m1 parameter to the boot strap.}
#'     \item{nsim}{The nsim parameter value used in the boot strap.}
#'     \item{time}{Time and date when boot strap was run.} 
#'    }
#'    
#' @examples
#' data(pbDat)
#' mod <- lm(y~x1 + x2 + g1, data=pbDat)
#' modNull <- lm(y~x2 + g1, data=pbDat)
#' pb <- pbnm(mod,modNull,seed=836928,test="F")
#' summary(pb)
#' 
#' @export 
#' 
summary.pbnm <- function (object,parm=NULL,...) {
  obj <- object
  sobj <- list()
  attr(sobj,"heading") <- "Summary of pbnm"
  class(sobj) <- "summary.pbnm"

  if (is.null(parm)) {
    if (obj$oneParm) {
      parm <- TRUE
    } else {
      parm <- FALSE
    }
  } else {
    if (!is.logical(parm)) stop("parm must be logical value")
    if (parm) {
      if (!obj$oneParm) {
        stop("No single parameter to report")
      }
    } else {
      if (!obj$nestStat) {
        stop("No nested model statistic to report")
      }
    }
  }

  if (parm) {
    dist <- obj$parmDist
    observed <- obj$parmObs
    name <- paste("abs(",obj$parmName,")",sep="")
  } else {
    dist <- obj$statDist
    observed <- obj$statObs
    name <- obj$statName
  }

  errSim <- (obj$errNull | obj$errAlt)
  fails <- sum(errSim)
  warnSim <- (obj$warnNull | obj$warnAlt)
  noEstSim <- (errSim | warnSim)
  noEst <- sum(noEstSim)
  extremSim <- (abs(dist[!noEstSim]) >= abs(observed))
  extrems <- sum(extremSim, na.rm=TRUE)
  nsim <- obj$nsim
  if (noEst == 0) {
    sobj$"P(>=obs)" <- extrems / nsim
  } else {
    sobj$"P(>=obs)" <- NA
    sobj$"#(>=obs)/runs" <- extrems / nsim
    sobj$"#(>=obs)/models" <- extrems / (nsim - noEst)
    sobj$"#(>=obs)+noEst/runs" <- (extrems + noEst) / nsim
  }
  sobj$noEst <- noEst
  sobj$fails <- fails
  sobj$errNull <- sum(obj$errNull,na.rm=TRUE)
  sobj$errAlt <- sum(obj$errAlt,na.rm=TRUE)
  sobj$warn <- sum(warnSim,na.rm=TRUE)
  sobj$warnNull <- sum(obj$warnNull,na.rm=TRUE)
  sobj$warnAlt <- sum(obj$warnAlt,na.rm=TRUE)
  sobj$name <- name
  sobj$observed <- observed
  sobj$testedDropped <- obj$testedDropped
  sobj$tested <- obj$tested
  sobj$Mod <- obj$Mod
  sobj$nsim <- obj$nsim
  sobj$time <- obj$time
  sobj
}

#' @export 
#' 
print.summary.pbnm <- function(x,...) {
  sobj <- x
  cat("Parametric bootstrap testing: ")
  if(sobj$testedDropped) {
    cat(paste(sobj$tested,sep="="), "= 0", "\n" )
  } else {
    cat(sobj$tested, "\n")
  }

  cat("from:",
      sub("^ *", "", deparse(sobj$Mod)), "\n" )

  cat(sobj$nsim, "samples were taken", sobj$time, "\n" )

  if (sobj$fail > 0) {
    cat(sobj$fail,"samples had errors,",
        sobj$errAlt,"in alternate model",
        sobj$errNull,"in null model \n"
    )
  }

  if (sobj$warn > 0) {
    cat(sobj$warn,"samples had warnings,",
        sobj$warnAlt,"in alternate model",
        sobj$warnNull,"in null model \n"
    )
  }

  if (sobj$noEst == 0) {
    cat("P(",sobj$name," > |", sobj$observed, "|) = ", sobj$"P(>=obs)", sep="" )
  } else {
    cat(sobj$noEst, " unused samples.  ",
        sobj$"#(>=obs)/runs",
        " <= P(",sobj$name," > |", sobj$observed, "|) <= ",
        sobj$"#(>=obs)+noEst/runs",
        sep=""
    )
  }
  cat("\n")
}


#' Confidence Interval for a parametric bootstrap of nested models
#' 
#' Empirical confidence intervals for a parametric bootstrap of
#' nested models
#' 
#' @inheritParams summary.pbnm
#' @param level Confidence level
#'   
#' @details Provides the confidence interval of the parameter or
#'   test statistic assuming the null model is the true model.
#'   This is different than what \code{confint(mod)} does,
#'   which is a confidence interval for the parameter or test
#'   statistic given the observed data.
#' 
#'   The fit errors are assumed to have values both negative
#'   infinity and positive infinity.
#'   This provides the most conservative interval when there
#'   are fit errors.
#'   
#'   The warnings are ignored in the calculations of the interval.
#' 
#' @return A data frame with columns giving
#'   lower bounds, upper bounds, number of simulations,
#'   number of simulations with fit failures, and the number
#'   of simulations with fit warnings.
#'   
#' @examples
#' data(pbDat)
#' mod <- lm(y~x1 + x2 + g1, data=pbDat)
#' modNull <- lm(y~x2 + g1, data=pbDat)
#' pb <- pbnm(mod,modNull,seed=836928,test="F")
#' confint(pb)
#' 
#' @export 
#' 
confint.pbnm <- function(object,parm=NULL,level=.95,...) {
  obj <- object
  if ( !is.numeric(level)) stop("lev must be numeric")
  if ( (level <= 0) | (level >= 1) ) stop("lev must be on (0,1)")

  if (is.null(parm)) {
    if (obj$oneParm) {
      parm <- TRUE
    } else {
      parm <- FALSE
    }
  } else {
    if (!is.logical(parm)) stop("parm must be logical value")
    if (parm) {
      if (!obj$oneParm) {
        stop("No single parameter to report")
      }
    } else {
      if (!obj$nestStat) {
        stop("No nested model statistic to report")
      }
    }
  }

  if (parm) {
    dist <- obj$parmDist
    name <- paste("abs(",obj$parmName,")",sep="")
    parmCI <- TRUE
  } else {
    dist <- obj$statDist
    name <- obj$statName
    parmCI <- FALSE
  }

  errSim <- (obj$errNull | obj$errAlt)
  fails <- sum(errSim)
  warnSim <- (obj$warnNull | obj$warnAlt)
  noEstSim <- (errSim | warnSim)
  noEst <- sum(noEstSim)

  ciObj <- matrix(c(NA,NA,NA,NA,NA),nrow=1)
  lower <- (1 - level) / 2
  upper <- 1 - lower
  rownames(ciObj) <- name
  colnames(ciObj) <- c(paste( c(as.character(100 * lower),
                                as.character(100 * upper)
                                ),
                              "%"
                            ),
                       "nsim",
                       "fails",
                       "warnings"
                       )
  lowerInd <- floor(obj$nsim * lower)
  if (noEst > 0) {
    if (lowerInd <= noEst) {
      lowerInd <- 0
    } else {
      lowerInd <- lowerInd - noEst
    }
  }
  if (lowerInd > 0) {
    sDist <- sort(dist[!noEstSim])
    ciObj[1,1] <- sDist[lowerInd]
    ciObj[1,2] <- sDist[(length(sDist) - lowerInd + 1)]
  } else {
    if (parmCI) {
      ciObj[1,1] <- -Inf
      if (grepl("\\|",name)) {
        if (!grepl("\\$",name)) {
          ciObj[1,1] <- 0
        }
      }
    } else {
      ciObj[1,1] <- 0
    }
    ciObj[1,2] <- Inf
  }
  ciObj <- as.data.frame(ciObj)
  ciObj[1,"nsim"] <- obj$nsim
  if (noEst > 0) {
    ciObj[1,"fails"] <- fails
  } else {
    ciObj <- ciObj[1,-which(colnames(ciObj) == "fails")]
  }
  warn <- sum( warnSim,na.rm=TRUE)
  if (warn > 0) {
    ciObj[1,"warnings"] <- warn
  } else {
    ciObj <- ciObj[1,-which(colnames(ciObj) == "warnings")]
  }
  ciObj
}

# CONSIDER ADJUSTING THE MARGINS IN THE GRAPH
#
#' Graphic Summary for a Parametric Bootstrap of Nested Models
#' 
#' Plots the empirical CDF from a parametric bootstrap of
#' nested models
#' 
#' @param x A "pbnm" object.
#' @param parm Set to \code{TRUE} to summarize the parmDist.
#'   Set to \code{FALSE} to summarize the statDist.
#' @param ... further arguments passed to or from other methods.
#'   
#' @details Simulated responses with fit errors are not included
#'   in the CDF.
#'   Simulated responses with fit warnings are included in the CDF.
#' 
#' @examples
#' data(pbDat)
#' mod <- lm(y~x1 + x2 + g1, data=pbDat)
#' modNull <- lm(y~x2 + g1, data=pbDat)
#' pb <- pbnm(mod,modNull,seed=836928,test="F")
#' plot(pb)
#' 
#' @export 
#' 
plot.pbnm <- function(x,parm=NULL,...) {
  obj <- x
  if (is.null(parm)) {
    if (obj$oneParm) {
      parm <- TRUE
    } else {
      parm <- FALSE
    }
  } else {
    if (!is.logical(parm)) stop("parm must be logical value")
    if (parm) {
      if (!obj$oneParm) {
        stop("No single parameter to report")
      }
    } else {
      if (!obj$nestStat) {
        stop("No nested model statistic to report")
      }
    }
  }

  if (parm) {
    dist <- obj$parmDist
    observed <- obj$parmObs
    name <- paste("abs(",obj$parmName,")",sep="")
  } else {
    dist <- obj$statDist
    observed <- obj$statObs
    name <- obj$statName
  }

  errSim <- (obj$errNull | obj$errAlt)
  warnSim <- (obj$warnNull | obj$warnAlt)
  noEstSim <- (errSim | warnSim)
  noEst <- sum(noEstSim)
  nsim <- obj$nsim
  est <- nsim - noEst
  sDist <- sort(dist[!noEstSim])
  if (length(sDist) < 2) {
    stop("Estimated distribution does not have enough members")
  }
  if (length(sDist) != est) {
    stop("Estimated distribution does not expected number of members")
  }
  Percentile <- (1:est) / est
  if (sDist[length(sDist)] < observed) {
    sDist <- c(sDist,observed)
    Percentile <- c(Percentile,1)
  } else if (sDist[1] > observed) {
    sDist <- c(observed,sDist)
    Percentile <- c(0,Percentile)
  }

  if (noEst > 0) {
    high <- (Percentile * est + noEst) / nsim
    low <- (Percentile * est) / nsim
    plot(sDist,high,
         type="l",
         xlab=name)
    lines(sDist,low)
  } else {
    plot(sDist,Percentile,
         type="l",
         xlab=name)
  }
  title("Nested Models Empirical CDF", line=3)
  title(deparse(obj$Mod[[2]]), line=2)
  abline(v=observed,col="red")
}
